<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\AdminNotification;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\City;
use App\Models\Country;
use App\Models\Property;
use App\Models\PropertyImage;
use Illuminate\Validation\ValidationException;

class PropertyController extends Controller
{
    public function myProperties()
    {
        $pageTitle = 'My Properties';
        $countries = Country::active()->get();
        $cities    = City::active()->get();
        $properties = Property::with('category', 'city', 'country')->where('user_id', auth()->user()->id)->searchable(['title'])->filter(['property_status', 'city_id', 'status'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.property.all', compact('pageTitle', 'cities', 'properties', 'countries'));
    }

    public function form()
    {
        $pageTitle  = 'Create Property';
        $countries = Country::active()->with('cities', function ($q) {
            $q->active();
        })->get();
        $cities  = City::active()->get();
        $categories = Category::active()->get();
        $activePlan = auth()->user()->activePlan;
        return view('Template::user.property.form', compact('pageTitle', 'cities', 'countries', 'categories', 'activePlan'));
    }

    public function edit($id)
    {
        $pageTitle  = 'Edit property';
        $countries  = Country::active()->with('cities', function ($q) {
            $q->active();
        })->get();
        $cities     = City::active()->get();
        $categories = Category::active()->get();
        $property   = Property::where('user_id', auth()->user()->id)->with('images')->findOrFail($id);

        $images = [];
        foreach ($property->images as $key => $image) {
            $img['id']  = $image->id;
            $img['src'] = getImage(getFilePath('property') . '/' . $image->name);
            $images[]   = $img;
        }
        $activePlan = auth()->user()->activePlan;
        return view('Template::user.property.form', compact('pageTitle', 'cities', 'countries', 'categories', 'property', 'images', 'activePlan'));
    }

    public function store(Request $request, $id = 0)
    {
        $this->validation($request, $id);

        if ($id) {
            $property = Property::findOrFail($id);
            $notification = 'Property updated successfully';
            $adminNotificationTitle = 'Property approval request';
        } else {
            $property = new Property();
            $notification = 'Property created successfully';
            $adminNotificationTitle = 'New property approval request';
        }

        if ($request->hasFile('display_image')) {
            try {
                $old = $property->display_image;
                $property->display_image = fileUploader($request->display_image, getFilePath('property'), getFileSize('property'), $old);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload your property image'];
                return back()->withNotify($notify);
            }
        }

        $user = auth()->user();
        $property->user_id          = $user->id;
        $property->property_id      = getTrx(8);
        $property->property_status  = $request->property_status;
        $property->title            = $request->title;
        $property->description      = $request->description;
        $property->category_id      = $request->category_id;
        $property->total_area       = $request->total_area;
        $property->country_id       = $request->country_id;
        $property->city_id          = $request->city_id;
        $property->location         = $request->location;
        $property->open_time        = $request->open_time;
        $property->close_time       = $request->close_time;
        $property->day_off          = $request->day_off ?? [];
        $property->price            = $request->price;
        $property->amenities        = $request->amenities ?? [];
        $property->tags             = $request->tags ?? [];
        $property->videos           = $request->videos ?? [];
        $property->status           = Status::PROPERTY_PENDING;
        $property->map              = $request->map;

        $property->save();
        $this->insertImages($request, $property, $id);

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = $user->id;
        $adminNotification->title = $adminNotificationTitle;
        $adminNotification->click_url = route('admin.property.details', $property->id);
        $adminNotification->save();

        $notify[] = ['success', $notification];
        return to_route('user.property.all')->withNotify($notify);
    }

    protected function validation($request, $id)
    {

        $imageRules  = $id ? 'nullable' : 'required';
        $user        = auth()->user();
        $plan        = $user->activePlan;

        if (!$plan) {
            throw ValidationException::withMessages(['error' => 'Your current plan has expired!']);
        }

        $propertyStatus = Status::BUY . "," . Status::RENT;

        $request->validate([
            'property_status'    => 'required|in:' . $propertyStatus,
            'title'              => 'required|string|max:255',
            'description'        => 'required|string',
            'category_id'        => 'required|exists:categories,id',
            'total_area'         => 'required|numeric',
            'price'              => 'required|numeric|gte:0',
            'country_id'         => 'required|exists:countries,id',
            'city_id'            => 'required|exists:cities,id',
            'location'           => 'required|string|max:255',
            'tags'               => 'nullable|array|min:1|max:10',
            'tags.*'             => 'required|string',
            'amenities'          => ['nullable', 'array', 'max:' . $plan->amenities],
            'amenities.*.name'   => 'required|string',
            'amenities.*.value'  => 'required|string',
            'day_off'            => 'nullable|array|min:1|max:7',
            'day_off.*'          => 'string|in:saturday,sunday,monday,tuesday,wednesday,thursday,friday',
            'open_time'          => 'nullable|date_format:H:i',
            'close_time'         => 'nullable|date_format:H:i',
            'display_image'      => [$imageRules, 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'images'             => ['nullable', 'array', 'max:' . $plan->images],
            'images*'            => ['nullable', 'images', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'videos'             => ['nullable', 'array', 'max:' . $plan->videos],
            'videos.*'           => 'required|url',
            'map'                => 'nullable|url',
        ], [
            'amenities.max'      => "Please follow maximum limit of amenities",
            'amenities.*.name'   => "Amenity name fields must not be empty",
            'amenities.*.value'  => "Amenity value fields must not be empty",
        ]);

        if (!$id) {

            $propertiesCount = Property::approved()->where('user_id', $user->id)->count();

            if ($plan->properties >= $propertiesCount) {
                $notify[] = ['error', 'The maximum amount of properties has already been added'];
                return to_route('pricing')->withNotify($notify);
            }
        }

        $oldImages = $request->old ?? [];
        $newImages = $request->images ?? [];

        $galleryImages = $oldImages + $newImages;

        if ($plan->images < count($galleryImages)) {
            throw ValidationException::withMessages(['error' => 'The maximum amount of image has already been uploaded']);
        }


        if (!$plan->allow_map && $request->map) {
            throw ValidationException::withMessages(['error' => 'You are not allowed to add map']);
        }
    }

    protected function insertImages($request, $property, $id)
    {
        $path = getFilePath('property');

        if ($id) {
            $this->removeImages($request, $property, $path);
        }

        $hasImages = $request->file('images');
        if ($hasImages) {
            $size = getFileSize('property');
            $images = [];

            if (!file_exists($path)) {
                mkdir($path, 0755, true);
            }

            foreach ($hasImages as $file) {
                try {
                    $name = fileUploader($file, $path, $size);
                    $image        = new PropertyImage();
                    $image->property_id = $property->id;
                    $image->name = $name;
                    $images[]     = $image;
                } catch (\Exception $exp) {
                    $notify[] = ['error', 'Couldn\'t upload property images'];
                    return back()->withNotify($notify);
                }
            }
            $property->images()->saveMany($images);
        }
    }

    protected function removeImages($request, $property, $path)
    {
        $previousImages = $property->images->pluck('id')->toArray();
        $imageToRemove  = array_values(array_diff($previousImages, $request->old ?? []));
        foreach ($imageToRemove as $item) {
            $propertyImage   = PropertyImage::find($item);
            @unlink($path . '/' . $propertyImage->name);
            $propertyImage->delete();
        }
    }

    public function switchStatus($id, $status)
    {
        $user     = auth()->user();
        $property = Property::where('user_id', $user->id)->where('id', $id)->firstOrFail();

        if ($status == Status::PROPERTY_DISABLED) {
            $notification = "Property disabled successfully";
        } elseif ($status == Status::PROPERTY_APPROVED) {
            $status       = Status::PROPERTY_PENDING;
            $notification = "Your approval request send successfully. Please wait for admin approval.";

            $adminNotification = new AdminNotification();
            $adminNotification->user_id = $user->id;
            $adminNotification->title = 'Property approval request';
            $adminNotification->click_url = route('admin.property.details', $property->id);
            $adminNotification->save();
        } else {
            abort(403);
        }

        $property->status = $status;
        $property->save();
        $notify[] = ['success',  $notification];
        return back()->withNotify($notify);
    }
}
