<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\City;
use App\Models\Country;
use App\Models\Page;
use App\Models\Plan;
use App\Models\Property;
use Illuminate\Http\Request;

class FrontController extends Controller
{
    public function properties(Request $request)
    {
        $pageTitle  = "Properties";
        $properties = Property::approved()
            ->whereHas('user.activePlan')
            ->with('category', 'city', 'country', 'user.activePlan')
            ->searchable(['tags'])
            ->filter(['category_id', 'country_id', 'city_id', 'property_status']);

        if ($request->min_price) {
            $properties->where('price', '>=', $request->min_price);
        }

        if ($request->max_price) {
            $properties->where('price', '<=', $request->max_price);
        }

        if ($request->area_min) {
            $properties->where('total_area', '>=', $request->area_min);
        }

        if ($request->area_max) {
            $properties->where('total_area', '<=', $request->area_max);
        }

        if ($request->location) {
            $properties->where('location', 'LIKE', '%' . $request->location . '%');
        }

        $properties = $properties->orderBy('id', 'DESC')->paginate(getPaginate(12));
        $categories = Category::active()->get();
        $cities     = City::active()->where('country_id', $request->country_id)->get();
        $countries  = Country::with('cities')->active()->get();
        return view('Template::properties', compact('pageTitle', 'properties', 'categories', 'cities', 'countries'));
    }

    public function propertyDetail($slug, $id)
    {
        $property = Property::approved()->with('category', 'city', 'country', 'images', 'user')->findOrFail($id);
        $plan     = $property->user->activePlan;
        if (!$plan) {
            abort(404);
        }

        $pageTitle          = 'Property Details';
        $categories         = Category::active()->hasProperties()->withCount('properties')->orderBy('properties_count', 'desc')->limit(10)->get();

        $totalProperties = Property::approved()->count();

        $seoContents['keywords']           = $property->tags ?? [];
        $seoContents['social_title']       = $property->title;
        $seoContents['description']        = strLimit(strip_tags($property->description), 200);
        $seoContents['social_description'] = strLimit(strip_tags($property->description), 150);
        $seoContents['image']              = getImage(getFilePath('property') . '/' . $property->display_image, getFileSize('property'));
        $seoContents['image_size']         = getFileSize('property');

        return view('Template::property_details',  compact('pageTitle', 'property', 'categories', 'plan', 'totalProperties', 'seoContents'));
    }

    public function pricingPlan()
    {
        $pageTitle = "Pricing Plan";
        $plans     = Plan::active()->orderBy('monthly_price', 'asc')->get();
        $sections  = Page::where('tempname', activeTemplate())->where('slug', 'pricing')->first();
        $seoContents = $sections->seo_content;
        $seoImage = @$seoContents->image ? getImage(getFilePath('seo') . '/' . @$seoContents->image, getFileSize('seo')) : null;
        return view('Template::plans', compact('pageTitle', 'plans', 'sections','seoContents','seoImage'));
    }
}
